<?php
// Inicia a sessão
session_start();
include 'conexao.php'; // Inclui a conexão com o banco de dados

// Configurações e Variáveis Iniciais
$usuario_id = (int)($_SESSION['usuario']['id'] ?? 0);
$perfil = $_SESSION['usuario']['perfil'] ?? '';
$ano_lectivo_atual = '2025/2026'; 

if (strtoupper(trim($perfil)) !== 'PROFESSOR' || $usuario_id <= 0) {
    header('Location: login.php');
    exit;
}

// Nome do professor logado (para a barra lateral)
$nome_professor_logado = '';
$sqlNome = "SELECT nome FROM usuarios WHERE id = ? LIMIT 1";
$stmt = $conn->prepare($sqlNome);
if ($stmt) {
    $stmt->bind_param("i", $usuario_id);
    $stmt->execute();
    $resNome = $stmt->get_result();
    if ($resNome && $resNome->num_rows === 1) {
        $rowNome = $resNome->fetch_assoc();
        $nome_professor_logado = $rowNome['nome'];
    }
    $stmt->close();
}

// ------------------------------------------
// LÓGICA DE SALVAR FALTAS (Com disciplina_id)
// ------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_faltas'])) {
    $turma_id = (int)($_POST['turma_id'] ?? 0);
    $disciplina_id = (int)($_POST['disciplina_id'] ?? 0); // Novo Campo
    $mes_selecionado = (int)($_POST['mes'] ?? 0);
    $ano = (int)date('Y'); 

    if ($turma_id > 0 && $disciplina_id > 0 && $mes_selecionado > 0) {
        $conn->begin_transaction();
        try {
            // 1. Apagar todas as faltas existentes para a TURMA, DISCIPLINA, mês e ano selecionados
            $data_inicio = date("$ano-$mes_selecionado-01");
            $data_fim = date("$ano-$mes_selecionado-t", strtotime($data_inicio));

             $sql_delete = "DELETE FROM faltas 
                           WHERE turma_id = ? 
                           AND disciplina_id = ? 
                           AND data_falta BETWEEN ? AND ?";
            $stmt_delete = $conn->prepare($sql_delete);
            $stmt_delete->bind_param("iiss", $turma_id, $disciplina_id, $data_inicio, $data_fim);
            $stmt_delete->execute();
            $stmt_delete->close();
            
            // 2. Inserir apenas as faltas que foram marcadas
            // Adicionado disciplina_id no INSERT
            $sql_insert = "INSERT INTO faltas 
                           (aluno_id, turma_id, disciplina_id, data_falta, falta, professor_id, ano_lectivo)
                           VALUES (?, ?, ?, ?, 1, ?, ?)";
            $stmt_insert = $conn->prepare($sql_insert);

            foreach ($_POST as $key => $value) {
                if (strpos($key, 'falta_') === 0) {
                    $parts = explode('_', $key);
                    $aluno_id = (int)$parts[1];
                    $data_falta = $parts[2]; 
                    
                    if ($aluno_id > 0 && $data_falta) {
                        $stmt_insert->bind_param(
                            "iiisis", // Tipos: i=int, s=string. 6 variáveis.
                            $aluno_id, $turma_id, $disciplina_id, $data_falta, $usuario_id, $ano_lectivo_atual
                        );
                        $stmt_insert->execute();
                    }
                }
            }
            $stmt_insert->close();
            
            $conn->commit();
            $_SESSION['mensagem_sucesso'] = "Faltas salvas com sucesso!";

        } catch (mysqli_sql_exception $e) {
            $conn->rollback();
            $_SESSION['mensagem_erro'] = "Erro ao salvar faltas: " . $e->getMessage();
        }
    } else {
        $_SESSION['mensagem_erro'] = "Selecione a Turma, a Disciplina e o Mês antes de salvar.";
    }
    header("Location: marcacao_faltas.php?turma_id={$turma_id}&disciplina_id={$disciplina_id}&mes={$mes_selecionado}");
    exit;
}

// ------------------------------------------
// LÓGICA DE CARREGAMENTO DE DADOS
// ------------------------------------------
$turma_id_selecionada = (int)($_GET['turma_id'] ?? 0);
$disciplina_id_selecionada = (int)($_GET['disciplina_id'] ?? 0); // Novo GET
$mes_selecionado = (int)($_GET['mes'] ?? date('n')); 

// 1. Turmas da escola (TODAS AS TURMAS)
$turmas = [];
$sqlTurmas = "
    SELECT id, nome, classe, sala, curso, turno
    FROM turmas
    ORDER BY classe, nome
";
$stmt = $conn->prepare($sqlTurmas);
$stmt->execute();
$turmas_res = $stmt->get_result();
while ($row = $turmas_res->fetch_assoc()) {
    $partes = [];
    if ($row['classe']) $partes[] = $row['classe'];
    if ($row['nome']) $partes[] = $row['nome'];
    if ($row['curso']) $partes[] = $row['curso'];
    $row['descricao_formatada'] = implode('-', $partes); 
    $turmas[] = $row;
}
$stmt->close();

// 2. Disciplinas da Turma Selecionada
$disciplinas = [];
if ($turma_id_selecionada > 0) {
    // Busca todas as disciplinas atribuídas à turma (via tabela atribuicoes)
    $sqlDisciplinas = "
        SELECT DISTINCT d.id, d.nome 
        FROM disciplinas d
        INNER JOIN atribuicoes a ON d.id = a.disciplina_id
        WHERE a.turma_id = ?
        ORDER BY d.nome
    ";
    $stmtDisc = $conn->prepare($sqlDisciplinas);
    $stmtDisc->bind_param("i", $turma_id_selecionada);
    $stmtDisc->execute();
    $disciplinas = $stmtDisc->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtDisc->close();
}


// 3. Alunos, faltas, dias úteis (se turma, disciplina e mês selecionados)
$alunos = [];
$faltas_marcadas = [];
$dias_uteis = [];
$nome_turma_selecionada = '';
$nome_disciplina_selecionada = '';
$classe_dinamica = '';
$curso_area_dinamica = '';

if ($turma_id_selecionada > 0 && $disciplina_id_selecionada > 0 && $mes_selecionado > 0) {
    // A. Informações da Turma e Disciplina
    foreach ($turmas as $turma) {
        if ($turma['id'] == $turma_id_selecionada) {
            $nome_turma_selecionada = $turma['descricao_formatada'];
            $classe_dinamica = $turma['classe'];
            $curso_area_dinamica = $turma['curso'];
            break;
        }
    }
    foreach ($disciplinas as $disciplina) {
        if ($disciplina['id'] == $disciplina_id_selecionada) {
            $nome_disciplina_selecionada = $disciplina['nome'];
            break;
        }
    }


    // B. Buscar alunos da turma (Lógica mantida)
    $sqlAlunos = "SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome";
    $stmtAlunos = $conn->prepare($sqlAlunos);
    $stmtAlunos->bind_param("i", $turma_id_selecionada);
    $stmtAlunos->execute();
    $alunos = $stmtAlunos->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtAlunos->close();

    // C. Gerar dias úteis (Lógica mantida)
    $ano = (int)date('Y');
    $num_dias = cal_days_in_month(CAL_GREGORIAN, $mes_selecionado, $ano);
    
    for ($dia = 1; $dia <= $num_dias; $dia++) {
        $data = new DateTime("$ano-$mes_selecionado-$dia");
        $dia_semana = (int)$data->format('N'); 
        
        if ($dia_semana >= 1 && $dia_semana <= 5) { 
            $dias_uteis[] = [
                'dia_numero' => $dia,
                'data_completa' => $data->format('Y-m-d'),
                'dia_semana_curto' => $data->format('D') 
            ];
        }
    }
    
    // D. Buscar faltas já marcadas (AGORA COM FILTRO DE DISCIPLINA)
    if (!empty($alunos) && !empty($dias_uteis)) {
        $data_inicio = $dias_uteis[0]['data_completa'];
        $data_fim = end($dias_uteis)['data_completa'];

        $sqlFaltas = "SELECT aluno_id, DATE_FORMAT(data_falta, '%Y-%m-%d') as data_falta
                      FROM faltas 
                      WHERE turma_id = ? 
                      AND disciplina_id = ? 
                      AND data_falta BETWEEN ? AND ? AND falta = 1";
        
        $stmtFaltas = $conn->prepare($sqlFaltas);
        // Novo bind_param: turma_id (i), disciplina_id (i), data_inicio (s), data_fim (s)
        $stmtFaltas->bind_param("iiss", $turma_id_selecionada, $disciplina_id_selecionada, $data_inicio, $data_fim);
        $stmtFaltas->execute();
        $resultFaltas = $stmtFaltas->get_result();
        
        while ($falta = $resultFaltas->fetch_assoc()) {
            $faltas_marcadas[$falta['aluno_id']][$falta['data_falta']] = true;
        }
        $stmtFaltas->close();
    }
}

// Mapas de Meses para exibição
$meses = [
    1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril', 
    5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto', 
    9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
];
$nome_mes = $meses[$mes_selecionado] ?? 'Mês Inválido';

// Mensagens
$mensagem_sucesso = $_SESSION['mensagem_sucesso'] ?? '';
$mensagem_erro = $_SESSION['mensagem_erro'] ?? '';
unset($_SESSION['mensagem_sucesso'], $_SESSION['mensagem_erro']);
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Marcação de Faltas (Por Disciplina)</title>
    <link rel="stylesheet" href="css/all.min.css">
    <script src="js/tailwindcss.js"></script>
<style>
    /* Estilos da tabela e layout */
    .faltas-table {
        border-collapse: collapse;
        width: 100%;
        font-size: 0.9rem;
    }
    .faltas-table th, .faltas-table td {
        border: 1px solid #ccc;
        padding: 2px;
        text-align: center;
    }

    /* 🔹 Coluna de nomes — agora mais estreita */
    .coluna-nome-aluno {
        min-width: 120px;  /* antes 150px */
        max-width: 160px;  /* antes 200px */
        text-align: left !important;
        padding-left: 6px !important;
        white-space: nowrap;
    }

    /* 🔹 Cabeçalho dos dias (rotacionado) */
    .rotate-col {
        height: 90px;
    }
    .rotate-col th {
        height: 100%;
        padding: 0;
        vertical-align: bottom;
        position: relative;
    }
    .dia-header {
        width: 30px;   /* antes 25px */
        height: 30px;  /* antes 25px */
        padding: 0;
        vertical-align: bottom;
        position: relative;
    }
    .dia-content {
        position: absolute;
        bottom: 0;
        left: 50%;
        transform-origin: left bottom;
        transform: rotate(-90deg) translateX(50%);
        white-space: nowrap;
        display: flex;
        flex-direction: column-reverse;
        align-items: center;
        height: 100%;
    }
    .dia-content .dia-semana {
        font-size: 0.6rem;
        margin-bottom: 2px;
    }
    .dia-content .dia-numero {
        font-size: 0.8rem;
        font-weight: bold;
    }

    /* 🔹 Célula e checkbox das faltas — agora maiores */
    .falta-display {
        width: 30px;   /* antes 25px */
        height: 30px;  /* quadrado visível */
        padding: 0;
    }
    .falta-checkbox {
        width: 25px;   /* antes 20px */
        height: 25px;  /* antes 20px */
        cursor: pointer;
        accent-color: red;
        display: block;
        margin: auto;
    }

    /* 🔹 Impressão */
    @media print {
        @page { size: A4 landscape; margin: 8mm; }
        .print-hidden, .sidebar, header, nav, .overlay, .form-filtros, .btn-salvar {
            display: none !important;
        }
        .faltas-table {
            font-size: 7pt;
        }
        .faltas-table th, .faltas-table td {
            padding: 1px 0px !important;
        }

        /* Coluna de nomes reduzida */
        .coluna-nome-aluno {
            min-width: 90px;
            max-width: 130px;
            padding-left: 2px !important;
        }

        /* Ajuste do cabeçalho e quadradinhos */
        .dia-header {
            width: 25px;
            height: 25px;
        }
        .falta-display {
            width: 25px;
            height: 25px;
        }

        /* Substitui checkbox por X ao imprimir */
        .faltas-table input[type="checkbox"] {
            display: none;
        }
        .falta-display:has(:checked)::before {
            content: "X";
            font-weight: bold;
            color: red;
            display: block;
            text-align: center;
            line-height: 1.2;
        }
    }
</style>

</head>
<body class="bg-gray-100 min-h-screen flex flex-col">

<header class="print-hidden fixed top-0 left-0 w-full md:left-64 md:w-[calc(100%-16rem)] h-16 bg-blue-900 text-white flex items-center justify-between p-4 shadow-lg z-50">
    <button id="menu-toggle" class="md:hidden text-2xl"><i class="fas fa-bars"></i></button>
    <span class="font-bold text-lg sm:text-xl">Marcação de Faltas (Por Disciplina)</span>
    <div class="flex items-center gap-4 ml-auto">
        <span class="hidden sm:block font-bold">Logado como: <?= htmlspecialchars($nome_professor_logado) ?></span>
    </div>
</header>

<div id="overlay" class="overlay"></div>

<nav id="sidebar" class="sidebar fixed top-0 left-0 w-64 h-full bg-blue-900 text-white p-4 pt-16 shadow-lg overflow-y-auto">
    <div class="flex flex-col items-center justify-center p-4">
        <span class="text-xl font-bold text-center">Utilizador: <?= htmlspecialchars($nome_professor_logado) ?></span>
    </div>
    <ul class="mt-8 space-y-2">
        <li><a href="painel_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-arrow-left"></i> <span>Voltar ao Painel</span></a></li>
        <li><a href="mini_pauta_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-pen"></i> <span>Lançar Notas (MT)</span></a></li>
        <li><a href="marcacao_faltas.php" class="flex items-center gap-3 p-3 rounded-lg bg-blue-700"><i class="fas fa-calendar-times"></i> <span>Marcar Faltas</span></a></li>
        <li><a href="login.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700"><i class="fas fa-sign-out-alt"></i> <span>Sair</span></a></li>
    </ul>
</nav>

<main class="content flex-grow p-4 md:p-6 md:ml-64 mt-16">
    <h1 class="text-3xl font-bold text-gray-800 mb-6 print-hidden">Marcação de Faltas</h1>

    <?php if ($mensagem_sucesso): ?>
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4 print-hidden"><?= htmlspecialchars($mensagem_sucesso) ?></div>
    <?php endif; ?>
    <?php if ($mensagem_erro): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4 print-hidden"><?= htmlspecialchars($mensagem_erro) ?></div>
    <?php endif; ?>

    <form action="marcacao_faltas.php" method="GET" class="form-filtros bg-white p-6 rounded-xl shadow-md mb-6 grid grid-cols-1 sm:grid-cols-4 gap-4 items-end print-hidden">
        
        <div>
            <label for="turma" class="block text-gray-700 font-semibold mb-2">Turma:</label>
            <select id="turma" name="turma_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <option value="">Selecione a Turma</option>
                <?php foreach ($turmas as $turma): ?>
                    <option value="<?= $turma['id'] ?>" <?= $turma_id_selecionada == $turma['id'] ? 'selected' : '' ?>><?= htmlspecialchars($turma['descricao_formatada']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div>
            <label for="disciplina" class="block text-gray-700 font-semibold mb-2">Disciplina:</label>
            <select id="disciplina" name="disciplina_id" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required <?= $turma_id_selecionada == 0 ? 'disabled' : '' ?>>
                <option value="">Selecione a Disciplina</option>
                <?php if ($turma_id_selecionada > 0): ?>
                    <?php foreach ($disciplinas as $disciplina): ?>
                        <option value="<?= $disciplina['id'] ?>" <?= $disciplina_id_selecionada == $disciplina['id'] ? 'selected' : '' ?>><?= htmlspecialchars($disciplina['nome']) ?></option>
                    <?php endforeach; ?>
                <?php endif; ?>
            </select>
        </div>

        <div>
            <label for="mes" class="block text-gray-700 font-semibold mb-2">Mês:</label>
            <select id="mes" name="mes" class="w-full p-3 rounded-lg border border-gray-300 focus:ring focus:ring-blue-200 transition-colors" required>
                <?php foreach ($meses as $num => $nome): ?>
                    <option value="<?= $num ?>" <?= $mes_selecionado == $num ? 'selected' : '' ?>><?= htmlspecialchars($nome) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div>
            <button type="submit" class="w-full bg-blue-900 text-white py-3 px-6 rounded-lg hover:bg-blue-800 transition"><i class="fas fa-filter"></i> Carregar</button>
        </div>
    </form>

    <?php if ($turma_id_selecionada > 0 && $disciplina_id_selecionada > 0 && $mes_selecionado > 0): ?>

        <div class="print-table-container bg-white p-4 md:p-6 rounded-xl shadow-md overflow-x-auto">
            
            <div class="print-header hidden">
                <h3 style="margin: 0; font-size: 11pt;">PAUTA DE FALTAS MENSAIS</h3>
                <p style="font-weight: bold;">TURMA: <?= htmlspecialchars($nome_turma_selecionada) ?> | DISCIPLINA: <?= htmlspecialchars($nome_disciplina_selecionada) ?></p>
                <p style="font-size: 9pt;">Classe: <?= htmlspecialchars($classe_dinamica) ?> | Mês: <?= htmlspecialchars($nome_mes) ?>/<?= date('Y') ?></p>
                <hr style="border: none; border-top: 1px solid #000; margin: 5px 0;">
            </div>
            
            <h2 class="text-xl font-bold text-gray-700 mb-4 print-hidden">
                Faltas: <?= htmlspecialchars($nome_turma_selecionada) ?> (<?= htmlspecialchars($nome_disciplina_selecionada) ?> - <?= htmlspecialchars($nome_mes) ?>/<?= date('Y') ?>)
            </h2>

            <?php if (empty($alunos)): ?>
                <p class="text-center text-gray-500 py-6">Nenhum aluno encontrado nesta turma.</p>
            <?php elseif (empty($dias_uteis)): ?>
                <p class="text-center text-gray-500 py-6">Nenhum dia útil encontrado para este mês.</p>
            <?php else: ?>
                
                <form method="POST">
                    <input type="hidden" name="turma_id" value="<?= $turma_id_selecionada ?>">
                    <input type="hidden" name="disciplina_id" value="<?= $disciplina_id_selecionada ?>">
                    <input type="hidden" name="mes" value="<?= $mes_selecionado ?>">

                    <table class="faltas-table bg-white">
                        <thead class="bg-gray-200">
                            <tr>
                                <th class="px-2 py-1" rowspan="2">Nº</th>
                                <th class="px-2 py-1 coluna-nome-aluno" rowspan="2">Nome do Aluno</th>
                                
                                <th class="rotate-col text-xs" colspan="<?= count($dias_uteis) ?>">
                                    <span style="display: block; transform: rotate(0deg); white-space: nowrap; margin-bottom: 5px;">DIAS ÚTEIS DO MÊS</span>
                                </th>
                                <th class="px-2 py-1" rowspan="2">Total Faltas</th>
                            </tr>
                            
                            <tr>
                                <?php foreach ($dias_uteis as $dia): ?>
                                    <th class="dia-header">
                                        <div class="dia-content">
                                            <span class="dia-semana"><?= substr(strtoupper($dia['dia_semana_curto']), 0, 1) ?></span>
                                            <span class="dia-numero"><?= $dia['dia_numero'] ?></span>
                                        </div>
                                    </th>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $i = 1; foreach ($alunos as $aluno):
                                $total_faltas = 0;
                            ?>
                                <tr>
                                    <td class="px-2 py-1 text-center"><?= $i++ ?></td>
                                    <td class="px-2 py-1 coluna-nome-aluno whitespace-nowrap"><?= htmlspecialchars($aluno['nome']) ?></td>
                                    
                                    <?php foreach ($dias_uteis as $dia):
                                        $data_completa = $dia['data_completa'];
                                        $input_name = "falta_{$aluno['id']}_{$data_completa}";
                                        $marcado = $faltas_marcadas[$aluno['id']][$data_completa] ?? false;
                                        if ($marcado) $total_faltas++;
                                    ?>
                                        <td class="falta-display">
                                            <input type="checkbox" 
                                                   name="<?= $input_name ?>" 
                                                   class="falta-checkbox"
                                                   value="1" 
                                                   <?= $marcado ? 'checked' : '' ?>>
                                        </td>
                                    <?php endforeach; ?>
                                    
                                    <td class="px-2 py-1 font-bold <?= $total_faltas > 0 ? 'text-red-600' : '' ?>">
                                        <?= $total_faltas ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <div class="mt-6 flex justify-between print-hidden">
                        <button type="submit" name="salvar_faltas" class="btn-salvar bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 transition shadow-lg"><i class="fas fa-save"></i> Salvar Faltas</button>
                        
                        <button type="button" onclick="window.print()" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition shadow-lg">
                            <i class="fas fa-print"></i> Imprimir Pauta de Faltas
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</main>

<script>
    document.getElementById("menu-toggle").addEventListener("click", function() {
        document.getElementById("sidebar").classList.toggle("active");
        document.getElementById("overlay").classList.toggle("active");
    });
    document.getElementById("overlay").addEventListener("click", function() {
        document.getElementById("sidebar").classList.remove("active");
        this.classList.remove("active");
    });
    
    // Script para recarregar as disciplinas quando a turma muda, sem submeter o formulário imediatamente
    document.getElementById('turma').addEventListener('change', function() {
        const turmaId = this.value;
        const currentDisciplinaId = document.getElementById('disciplina').value;
        const currentMes = document.getElementById('mes').value;
        
        // Se a turma for alterada, recarrega a página apenas com a nova turma e o mês atual
        // e limpa a disciplina para forçar a seleção.
        if (turmaId) {
             window.location.href = `marcacao_faltas.php?turma_id=${turmaId}&mes=${currentMes}`;
        } else {
             window.location.href = `marcacao_faltas.php`;
        }
    });

    // Submete o formulário se Turma, Disciplina e Mês estiverem selecionados (opcional)
    document.getElementById('disciplina').addEventListener('change', function() {
         if (document.getElementById('turma').value && document.getElementById('mes').value) {
            this.closest('form').submit();
         }
    });
    document.getElementById('mes').addEventListener('change', function() {
         if (document.getElementById('turma').value && document.getElementById('disciplina').value) {
            this.closest('form').submit();
         }
    });

    // Se houver uma turma selecionada, garante que o select de disciplina está habilitado
    window.onload = function() {
        if (document.getElementById('turma').value > 0) {
            document.getElementById('disciplina').disabled = false;
        }
    }
</script>
</body>
</html>